import os
import yaml
import numpy as np
import pandas as pd
import argparse


def make_kernels_from_eigs(eigs_path: str, output_dir: str, cfg: dict) -> None:
    """
    Generate per-L kernel arrays from a list of eigenvalues (rho).

    The input CSV should contain a column labelled ``rho``.
    For each target L, three kernels are generated:

    * ``kernel_U1_L{L}.npy`` contains the scalar eigenvalues (resampled to 2*L*L entries).
    * ``kernel_SU2_L{L}.npy`` contains 2×2 matrices rho * σ_z / 2.
    * ``kernel_SU3_L{L}.npy`` contains 3×3 matrices rho * λ_3 / 2.

    The output files are written into ``output_dir``. Directories are created if missing.
    Any existing files with the same names will be overwritten.
    """

    os.makedirs(output_dir, exist_ok=True)

    # Load eigenvalues
    df = pd.read_csv(eigs_path)
    if 'rho' not in df.columns:
        raise ValueError(f"Expected 'rho' column in {eigs_path}")

    rho_master = df['rho'].to_numpy(dtype=float)

    # Determine L list from config or fallback
    L_list = cfg.get("L_values") or cfg.get("L_list") or [6, 8, 10, 12, 16, 20]
    print(f"Building kernels for L values: {L_list}")

    # Define SU2 and SU3 diagonal generators
    sigma_z = np.array([[1.0, 0.0], [0.0, -1.0]], dtype=float)
    lambda3 = np.array([[1.0, 0.0, 0.0],
                        [0.0, -1.0, 0.0],
                        [0.0,  0.0,  0.0]], dtype=float)

    # Resample for each L and save
    for L in L_list:
        N = 2 * L * L
        # Resample rho to length N
        x_old = np.linspace(0, 1, len(rho_master))
        x_new = np.linspace(0, 1, N)
        rho = np.interp(x_new, x_old, rho_master)

        # Save U(1) kernel
        u1_path = os.path.join(output_dir, f'kernel_U1_L{L}.npy')
        np.save(u1_path, rho)

        # Save SU(2) kernel
        su2_path = os.path.join(output_dir, f'kernel_SU2_L{L}.npy')
        np.save(su2_path, (rho[:, np.newaxis, np.newaxis] * sigma_z) / 2.0)

        # Save SU(3) kernel
        su3_path = os.path.join(output_dir, f'kernel_SU3_L{L}.npy')
        np.save(su3_path, (rho[:, np.newaxis, np.newaxis] * lambda3) / 2.0)

        print(f"  L={L}: saved {u1_path}, {su2_path}, {su3_path}")

    print(f"All kernels saved to {output_dir}")


def main() -> None:
    parser = argparse.ArgumentParser(description="Build per-L kernels from eigenvalues")
    parser.add_argument('--config', required=True, help='Path to YAML configuration file')
    parser.add_argument('--eigs', required=True, help='Path to CSV file of eigenvalues')
    parser.add_argument('--output-dir', required=True, help='Directory to write kernels')
    args = parser.parse_args()

    # Load config
    with open(args.config, "r", encoding="utf-8") as f:
        cfg = yaml.safe_load(f)

    make_kernels_from_eigs(args.eigs, args.output_dir, cfg)


if __name__ == '__main__':
    main()
